extensions [ gis ]

globals ; define here if there is only one value for entire landscape (applies to all patches and tortoises)
  [
  gt_dataset
  boundary_dataset
  ranges_dataset
  roads_dataset
  streams_dataset
  HA-per-cell
  month 
  year 
  activity-season
;  egg-to-juvenile-mortality
  min-juvenile-age
  min-adult-age
;  adult-annual-mortality
;  juvenile-annual-mortality
  juvenile-death-prob-per-month
  adult-death-prob-per-month 
  adult-dispersal-prob
  adult-dispersal-prob-per-month
  juvenile-dispersal-prob
  juvenile-dispersal-prob-per-month
;  eggs-per-female
;  female-reproduction-probability
;  max_age
  total-GT 
  total-adults 
  total-juveniles 
  total-patches
  average-GT-density-all 
  average-GT-density-GTMA
  Remaining-indiv-to-release
  release-year?  
  habitat-unsuitable           ; ( adult-carrying-capacity < 1.125 )
  habitat-low                  ; ( adult-carrying-capacity = 1.125 and adult-carrying-capacity < 6.75 )
  habitat-moderate             ; ( adult-carrying-capacity = 6.75 and adult-carrying-capacity < 11.25 )
  habitat-high                 ; ( adult-carrying-capacity => 11.25 )
  patches-in-study-area
  display-GT-mgmt-areas
  ]

patches-own    ; define here if it varies with each patch
  [ 
  study_area                   ; 1 if within study area, 0 otherwise
  projected-carrying-capacity  ; GT carrying capacity (tortoises/ha)for adults; resampled - 105.6 GT/cell max; 0 min.; read directly from map; estimated from Guyer formula
  adult-carrying-capacity      ; allows user to modify carrying capacity from map based on knowledge of real on-the-ground conditions; default value is 1 (which retains value from map) 
  gt_management_area           ; 1 if within an identified gopher tortoise management area (GTMA), 0 otherwise; resampled - 25 pixels per cell
  soils                        ; 1-Priority, 2-Suitable, 3-Marginal, 0-not suitable
  ranges                       ; 1 if within a firing range, 0 otherwise; resampled - 25 pixels per cell
  wetlands                     ; 1 if wetland, 0 otherwise; resampled - 25 pixels per cell
  tmp                          
  habitat-quality              ; scale of 0-1 and correlated with carrying capacity  
  eggs-hatchlings
  GT-space-available           ; carrying capacity minus # adults currently occupying a cell
  habitat-class
  viable
  neighborhood-viability
  ]

breed [ GT a-GT ]   ; define here if it varies with each tortoise
GT-own [
  sex
  age
  step
  max-steps
]

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; SETUP 
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to setup-fts-current
  clear-all

  ; Set all global variables
  set year 1
  set min-juvenile-age 1
  set min-adult-age  15
  set adult-dispersal-prob 2     ; annual dispersal probability for resident adults (%)
  set juvenile-dispersal-prob 10   ; annual dispersal probability for resident juveniles (%)
   
  ; convert annual death rates to monthly death rates
  ; calculated here in case user adjusts the rates during a run 
  ; based on assumption that mortality can occur in any month and is equal across months     
  set adult-death-prob-per-month 100 * ( 1 - e ^ (ln (1 - (adult-annual-mortality / 100)) / 12))
  set juvenile-death-prob-per-month 100 * ( 1 - e ^ (ln (1 - (juvenile-annual-mortality / 100)) / 12))
  
  ; convert annual dispersal rates to monthly dispersal rates
  ; calculated here in case user adjusts the rates during a run
  ; calculations based on assumption that dispersal probability is equal across all months in which movement occurs
  set adult-dispersal-prob-per-month 100 * ( 1 - e ^ (ln (1 - (adult-dispersal-prob / 100)) / 7))
  set juvenile-dispersal-prob-per-month 100 * ( 1 - e ^ (ln (1 - (juvenile-dispersal-prob / 100)) / 7))
            
  set HA-per-cell ( 150 * 150 ) / 10000  
  read-maps
  create-tort  
  setup-plot
  set patches-in-study-area patches with [ adult-carrying-capacity != -1 and study_area != 0 ]
  reset-ticks
end

to create-tort  
 create-GT initial-pop-size
 [
  setxy random-xcor random-ycor
  initialize-GT ( abs ( random-normal 0 30 ) )  ;  absolute value of a one-tailed normal distribution and SD of 30
  while [ gt_management_area < 13 OR adult-carrying-capacity < 1 OR ranges > 13]  ; place tortoises in cells where GTMA is >50% of cell (since range of possible values is 0-25, 13 is >50%), carrying capacity is greater than 1, and ranges are less than 50% of the cell.
  [ setxy random-xcor random-ycor ]
 ]
end

to initialize-GT [ startage ]    
  set shape "turtle"
  set age startage
  ifelse random 2 = 0 [ set sex "F" set color 15 ] [ set sex "M" set color 95 ] 
  set label-color blue - 2
  if age >= min-adult-age [ set size 2 ] 
  if age < min-adult-age [ set size 1 ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; GO
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to go                        
  if not any? GT [stop]
  set month month + 1
  if month > 12 [ set month 1 set year year + 1 ]
  tick
  
  if ( month > 3 and month < 11 ) 
  [ 
  ask GT 
    [
     set step 0
     set max-steps 8
    ] 
  ask patches-in-study-area [ move-tortoises ] 
  ]
    
  if month = 5
   [ ask GT [search-for-habitat] ]
 
  if ( month > 3 and month < 11 )
  [ ask GT [ disperse ] ]
   
  if ( month = 6 )
  [ ask GT with [ ( sex = "F" ) and ( age > min-adult-age )] [ reproduce ] ]
  
  if ( month = 12)
  [ ask GT [ change-age ] ]
    
  ask GT [ potentially-die ]
  
  if month = 9
  [ ask patches-in-study-area [ process-eggs-hatchlings ] ]
  
  do-plot
 end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; PATCH PROCEDURES 
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
 
to move-tortoises   
 ; tortoises will not be forced to move-from-patch as long as the number of adults in the patch is less than or equal to adult carrying capacity of the patch.
 ; If the number of adults in a patch exceeds the carrying capacity of the patch, the youngest adults will be forced to move-from-patch
 ; This procedure repeats until the number of adult tortoises inside the patch is at or below carrying capacity
 ; The procedure below (to move-tortoises) is a patch procedure (all tortoises don't move)
    ; Set the max steps each GT can take this time step (tick)
                    
  let adults GT-here with [ age >= min-adult-age ]  ; adults are individuals of age 15 and over
  while [ count adults > adult-carrying-capacity AND ( [max-steps - step] of min-one-of adults [ age ] > 0) ]  ; while there are too many adults on this patch (here)
  [ 
    ask ( min-one-of adults [ age ] ) ; youngest adult is selected, if same age, randomly select between youngest
    [
      move-from-patch
    ]  
    set adults GT-here with [ age >= min-adult-age ] ; count the adults here
  ]
end

to process-eggs-hatchlings ; patch procedure to create new tortoises from their stock of eggs-hatchlings
  while [ eggs-hatchlings > 0 ] [
      if ( random 100 > egg-to-juvenile-mortality ) [
        sprout-GT 1 [ initialize-GT 1 ] 
      ]
    set eggs-hatchlings eggs-hatchlings - 1
    ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; TORTOISE PROCEDURES 
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
     
to move-from-patch  ; single tortoise procedure; called only when patch adult-carrying capacity is exceeded; then, youngest adult forced to move
    while [ ( count GT-here with [ age >= min-adult-age ] > adult-carrying-capacity ) ] 
   [
    ask neighbors [ set GT-space-available ( adult-carrying-capacity - count ( GT-here with [ age >= min-adult-age ] ) )]  ; Look at surrounding 8 patches to find the patch with the most space available
    let winner max-one-of neighbors [ GT-space-available ]     ; choose one of the patches with the most GT space available. If there is a tie, a winner is randomly selected by process of the max-one-of syntax.
    if ( winner = nobody )   
    [    ; If there is no one with space available, move to a random neighbor where wetlands don't occupy entire (=25) cell. Forces GT to move through unsuitable habitat if cc=0 to keep looking for multiple steps.
      ifelse ( any? neighbors with [ wetlands < 25 ]) 
      [ set winner one-of neighbors with [ wetlands < 25 ] ]
      [ die ] 
    ] 
    move-to winner
    set step step + 1
    if ( step > max-steps )[ stop ] ; tortoises stop if they can not find a patch in allowable steps     
   ]
end

to search-for-habitat  ; single tortoise procedure; tortoise periodically assess habitat of their current patch and compare to surrounding patches; will move to better patch if space is available
  ask neighbors [ set GT-space-available ( adult-carrying-capacity - count ( GT-here with [ age >= min-adult-age ] ) )]  ; Look at surrounding 8 patches to find the patch with the most space available
  let short-list neighbors with [ GT-space-available > 0 ]   ; Find out which neighbors have space available
  let winner max-one-of short-list [ habitat-quality ]     ; Pick the one out of those neighbors with space available that has the highest quality habitat
     ; let winner max-one-of ( neighbors with [ GT-space-available > 0 ] ) [habitat-quality]     ;  THIS LINE IS A COMBO OF THE ABOVE 2 LINES. choose one of the patches with the most GT space available and highest habitat quality
  if (winner != nobody) ; if the winner doesn't = nobody, move to that winner
  [
    if ( [habitat-quality] of winner ) > habitat-quality [ move-to winner ]
  ] 
end
 
to disperse  
  ; every month of activity season there is a small probability that a resident tortoise will disperse due to factors unrelated to habitat quality or carrying capacity
  
  if ( age >= min-adult-age ) AND ( random-float 100 < adult-dispersal-prob-per-month )
    [ right random 360
      forward 1
      disperse-from-patch ]
  if ( age < min-adult-age ) AND ( random-float 100 < juvenile-dispersal-prob-per-month )
    [ right random 360
      forward 1
      disperse-from-patch ]
end

to disperse-from-patch    
  set step 0
  set max-steps  8 
  
  while [ count GT-here with [ age >= min-adult-age ] < 2 ]  ; while you are the only tortoise in the patch
  [ 
    forward 1
    if ( wetlands = 25 )  ; if the entire cell is wetland habitat, turn around and move back to your original cell
    [
      right 180
      forward 1
      stop
    ]
    set step step + 1
    if ( step > max-steps )[ stop ] ; tortoises stop if they can not find a patch in allowable steps     
  ]
end

to reproduce 
  let reproduction-probability female-reproduction-probability 
  let clutch-size eggs-per-female 
  if habitat-class = 0 [ set clutch-size 0 ]  
  if habitat-class = 1 [ set clutch-size (eggs-per-female - 2) ]
  
  if ( reproduction-probability  > random 100 )
  [ set eggs-hatchlings eggs-hatchlings + clutch-size ]
end

to change-age
  set age age + 1
  if age >= min-adult-age [ set size 2 ]
end
 
to potentially-die
    ifelse ( age < min-adult-age )
    [ if ( random-float 100 < ( juvenile-death-prob-per-month * death-modifier-juv) ) [ die ] ] ;"random-float" uses decimal places and therefore has greater accuracy than "random"
    [ if ( random-float 100 < ( adult-death-prob-per-month * death-modifier-adult ) ) [ die ] ]
    if month = 11 and habitat-quality = 0 [ die ]  ; tortoises can not overwinter in unsuitable habitat
    if age = max_age [ die ] ; tortoises can not live past the maximum age set through behavior space.
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; PLOTTING PROCEDURES
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to setup-plot ;; set up plotting
  set-current-plot "Number of Tortoises"
  set-plot-y-range 0 count GT
  
  set-current-plot "Average Tortoise Density"
  set-plot-y-range 0 2

  set-current-plot "Age Distribution"
  set-plot-x-range 0 110
  set-plot-y-range 0 count GT
  set-histogram-num-bars 10
 end

to do-plot
;  if month = 6       
;[ 
  set total-GT count GT
  set total-patches count patches
  set total-adults  ( count GT  with [ age > min-adult-age - 1 ] )
  set total-juveniles ( count GT with [ age < min-adult-age ] )
  set average-GT-density-all (( total-GT / total-patches ) / HA-per-cell ) 
  set average-GT-density-GTMA ((( count GT with [ gt_management_area > 13 ] ) / ( count patches with [ gt_management_area > 13 ] )) / HA-per-cell )       

  set-current-plot "Number of Tortoises"
  set-current-plot-pen "total" 
    plot total-GT
  set-current-plot-pen "adults"  
    plot total-adults
  set-current-plot-pen "juveniles"  
    plot total-juveniles
    
  set-current-plot "Average Tortoise Density"
  set-current-plot-pen "All Patches" 
   plot average-GT-density-all
  set-current-plot-pen "All GTMA"  
    plot average-GT-density-GTMA 
  
  set-current-plot "Age Distribution"
  histogram [ age ] of GT         ; using the default plot pen
;]
end

to save-final-info
  save-total-tort-map ( word "total-tort-" Initial-pop-size "-" substring date-and-time 0 2 "-" substring date-and-time 3 5 "-"  substring date-and-time 6 8  "-" substring date-and-time 16 27 ".asc" )
  save-total-adult-map ( word "total-adul-" Initial-pop-size "-" substring date-and-time 0 2 "-" substring date-and-time 3 5 "-"  substring date-and-time 6 8  "-" substring date-and-time 16 27 ".asc" )
  save-total-females-map ( word "total-fema-" Initial-pop-size "-" substring date-and-time 0 2 "-" substring date-and-time 3 5 "-"  substring date-and-time 6 8  "-" substring date-and-time 16 27 ".asc" )
  save-age-structure-to-file
end

to save-total-tort-map [ file-name ]
  ask patches [ 
    ifelse study_area = 0
      [ set tmp -1 ]
      [ set tmp count GT-here ]     ;;outputs # tortoises/cell
  ]
  print file-name
  gis:store-dataset ( gis:patch-dataset tmp ) file-name
end

to save-total-adult-map [ file-name ]
  ask patches [ 
    let adults GT-here with [ age >= min-adult-age ]
    ifelse study_area = 0
      [ set tmp -1 ]
      [ set tmp count adults ]     ;;outputs # tortoises/cell
  ]
  print file-name
  gis:store-dataset ( gis:patch-dataset tmp ) file-name
end

to save-total-females-map [ file-name ]
  ask patches [ 
    ifelse study_area = 0
      [ set tmp -1 ]
      [ set tmp count GT with [ sex = "F" ] ]     ;;outputs # tortoises/cell
  ]
  print file-name
  gis:store-dataset ( gis:patch-dataset tmp ) file-name
end

to save-age-structure-to-file
  file-open "adult-mort-initial-20.csv"
  file-print ( word
   count GT with [ age < 1 ]                  ","
   count GT with [ age >=  1 AND age < 11 ]   ","
   count GT with [ age >= 11 AND age < 21 ]   ","
   count GT with [ age >= 21 AND age < 31 ]   ","
   count GT with [ age >= 31 AND age < 41 ]   ","
   count GT with [ age >= 41 AND age < 51 ]   ","
   count GT with [ age >= 51 AND age < 61 ]   ","
   count GT with [ age >= 61 AND age < 71 ]   ","
   count GT with [ age >= 71 AND age < 81 ]   ","
   count GT with [ age >= 81 AND age < 91 ]   ","
   count GT with [ age >= 91 AND age < 100 ]  ","
   count GT with [ age >= 100 ]
  )
  file-close
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
; READ MAPS
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to read-maps
  setup-GIS
  ask patches with [ pxcor = max-pxcor ] [ set wetlands 25 ]   ; sets world view as an island so that GT cannot get stuck at view boundaries.
  ask patches with [ pycor = max-pycor ] [ set wetlands 25 ]
  ask patches with [ pxcor = min-pxcor ] [ set wetlands 25 ]
  ask patches with [ pycor = min-pycor ] [ set wetlands 25 ]
  
    ;  need to create patch variable that uses info for both map estimate of CC and modifier selected by user
  ask patches [ set adult-carrying-capacity projected-carrying-capacity * carrying-capacity-modifier ]
  let max-val [ adult-carrying-capacity ] of max-one-of patches [ adult-carrying-capacity ]  ;;max-val established as a local variable that represents the maximum carrying capacity value across the landscape. This allows us to use habitat quality as a proxy that ranges from 0-1.
  ask patches 
  [ ifelse adult-carrying-capacity > 0 
    [ set habitat-quality adult-carrying-capacity / max-val ]
    [ set habitat-quality 0 ]
  ]
  ask patches [ set tmp adult-carrying-capacity ]  display-tmp
  set display-GT-mgmt-areas FALSE
  
  set habitat-unsuitable           0
  set habitat-low                  1
  set habitat-moderate             2
  set habitat-high                 3

  ask patches      ; ranking habitat based on carrying capacity scale and the interface modifier.
  [         
   if adult-carrying-capacity < 1.125 [ set habitat-class habitat-unsuitable ]
   if adult-carrying-capacity >=  1.125 and adult-carrying-capacity < 6.75 [ set habitat-class habitat-low ]
   if adult-carrying-capacity >= 6.75 and adult-carrying-capacity < 11.25 [ set habitat-class habitat-moderate ]
   if adult-carrying-capacity >= 11.25 [ set habitat-class habitat-high ]
  ]
  display-vector-maps
  display-raster-maps
end

to setup-GIS
  ; Establish coordinate transformation between GIS maps and Netlogo patches
  let dataset gis:load-dataset "maps/rast/boundary.asc"                                            ; Read the boundary map
  gis:set-transformation (gis:envelope-of dataset) (list min-pxcor max-pxcor min-pycor max-pycor)  ; Establish transformation
  ; Print a warning to the user to ensure that the display area matches the map
;  type "Make sure that the display area is setup for "
;  type gis:width-of dataset
;  type " columns and "
;  type gis:height-of dataset
;  print " rows."
  gis:apply-raster dataset study_area

  ; Read in remaining raster maps into patch variables
  gis:apply-raster gis:load-dataset "maps/rast/gt_per_150m.asc"   projected-carrying-capacity
  gis:apply-raster gis:load-dataset "maps/rast/gt_mgmt_areas.asc" gt_management_area
  ; gis:apply-raster gis:load-dataset "maps/rast/gt_soils.asc"      soils
  ; gis:apply-raster gis:load-dataset "maps/rast/ranges.asc"        ranges
  gis:apply-raster gis:load-dataset "maps/rast/wetlands.asc"      wetlands
  
  ; Read in vector maps
  ;set gt_dataset       gis:load-dataset "maps/vect/gopher_tortoise/gopher_tortoise_cropped.shp"
  ;set boundary_dataset gis:load-dataset "maps/vect/boundary/boundary_cropped.shp"
  ;set ranges_dataset   gis:load-dataset "maps/vect/ranges/ranges_cropped.shp"
  ;set roads_dataset    gis:load-dataset "maps/vect/roads/roads_cropped.shp"
  set streams_dataset  gis:load-dataset "maps/vect/streams/streams_cropped.shp"
end

to display-tmp
  ; Display the cc values as a grey-scale background
  let max-val [ tmp ] of max-one-of patches [ tmp ]
  let min-val [ tmp ] of min-one-of patches [ tmp ]
  let range max-val - min-val
  ask patches
    [ set pcolor 9.9 - ( 9.9 * (tmp - min-val) / range )]
end

to display-vector-maps    
  ; Display the tortoise management area boundaries as green lines
  ;gis:set-drawing-color 57      gis:draw gt_dataset 3
  ;gis:set-drawing-color 3      gis:draw boundary_dataset 4
  ;gis:set-drawing-color 7      gis:draw roads_dataset 1
  gis:set-drawing-color 98       gis:draw streams_dataset 1
  ;gis:set-drawing-color red        gis:draw ranges_dataset 3
end

to display-raster-maps    
  ; Display the tortoise management area boundaries as green lines
  ;gis:set-drawing-color 57      gis:draw gt_dataset 3
  ;gis:set-drawing-color 3      gis:draw boundary_dataset 4
  ;gis:set-drawing-color red        gis:draw ranges_dataset 1
  ;gis:set-drawing-color 7        gis:draw roads_dataset 1
  gis:set-drawing-color 98       gis:draw streams_dataset 1
end

;to-report max_age
;  report count GT WITH [ age > 100 ]
;end





@#$#@#$#@
GRAPHICS-WINDOW
643
28
2153
899
-1
-1
4.0
1
10
1
1
1
0
0
0
1
0
374
0
209
0
0
1
months
30.0

TEXTBOX
34
10
247
53
Initial Tortoise Abundance
14
0.0
1

TEXTBOX
31
287
201
321
Landscape Initialization
14
0.0
1

BUTTON
33
310
181
343
Load Maps
setup-fts-current
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
382
11
478
61
STEP
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
494
11
587
62
GO
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
245
71
608
238
Number of tortoises
Time (years)
Individuals
0.0
200.0
0.0
1000.0
true
true
"" ""
PENS
"total" 1.0 0 -16777216 true "" ""
"adults" 1.0 0 -2674135 true "" ""
"juveniles" 1.0 0 -13345367 true "" ""

PLOT
244
237
609
403
Average tortoise density
Time (years)
Individuals / ha
0.0
200.0
0.0
2.0
true
true
"" ""
PENS
"All Patches" 1.0 0 -10899396 true "" ""
"All GTMA" 1.0 0 -955883 true "" ""

SLIDER
11
27
214
60
Initial-pop-size
Initial-pop-size
2000
10000
3000
500
1
NIL
HORIZONTAL

PLOT
245
401
610
551
Age Distribution
Years
Individuals
0.0
110.0
0.0
1000.0
true
false
"" ""
PENS
"default" 5.0 1 -16777216 true "" ""

MONITOR
12
416
69
461
NIL
Year
17
1
11

MONITOR
79
416
136
461
NIL
Month
17
1
11

MONITOR
148
416
211
461
NIL
count GT
0
1
11

BUTTON
35
347
181
380
Toggle View GTMA
ifelse (display-GT-mgmt-areas)\n[  ask patches [ set tmp adult-carrying-capacity ]  display-tmp\n  set display-GT-mgmt-areas FALSE ]\n[  ask patches [ set tmp gt_management_area ]\n  set display-GT-mgmt-areas TRUE ]\n\ndisplay-tmp
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
166
211
199
Carrying-capacity-modifier
Carrying-capacity-modifier
0.1
1.0
1
0.1
1
NIL
HORIZONTAL

SLIDER
11
204
211
237
death-modifier-adult
death-modifier-adult
.5
1
1
0.1
1
NIL
HORIZONTAL

SLIDER
11
241
211
274
death-modifier-juv
death-modifier-juv
0.5
1
0.9
0.1
1
NIL
HORIZONTAL

TEXTBOX
72
142
222
160
Modifiers
14
0.0
1

SLIDER
11
64
213
97
Adult-annual-mortality
Adult-annual-mortality
1.5
12
1.5
1.5
1
NIL
HORIZONTAL

SLIDER
11
97
213
130
Juvenile-annual-mortality
Juvenile-annual-mortality
10
30
15
5
1
NIL
HORIZONTAL

SLIDER
6
475
223
508
Egg-to-juvenile-mortality
Egg-to-juvenile-mortality
90
96
90
1
1
NIL
HORIZONTAL

SLIDER
7
507
224
540
Female-reproduction-probability
Female-reproduction-probability
60
100
60
10
1
NIL
HORIZONTAL

SLIDER
6
540
224
573
Max_age
Max_age
60
100
60
10
1
NIL
HORIZONTAL

SLIDER
6
572
224
605
Eggs-per-female
Eggs-per-female
4
8
4
1
1
NIL
HORIZONTAL

@#$#@#$#@
## Purpose
The was to assess the demographic factors most likely to influence persistence and abundance of gopher tortoises across Fort Stewart’s range lands. However, the most influential (i.e., “sensitive”) parameters were not necessarily the ones most amenable to manipulation, but some demographic traits were known to be influenced by habitat quality and thus could potentially serve as management targets. Ultimately, the purpose of the model was to guide management of gopher tortoises at the installation level by determining whether current habitat conditions and management practices would be likely to ensure the species’ continued persistence on the installation, and if not, to evaluate whether increased habitat manipulation would be likely to influence demographic parameters sufficiently to improve overall species viability. 

## Spatial and temporal scale
The spatial extent of the model’s landscape is Fort Stewart, Georgia. Each patch corresponds to a 150 x 150 m (2.25 ha) area. This size was large enough to model the entire Fort Stewart landscape and for each patch to support more than one tortoise, yet not so large as to preclude movement among patches by individual tortoises during their lifetime. Each time step in the model represented one month to allow movement of individual tortoises between patches and to capture seasonal differences in movement probability (active vs. inactive season). Data were output from the model at 20-year intervals over the course of 100-year simulations. 

## Patch variables
The primary patch variable was carrying capacity which is the number of adult tortoises each patch is predicted to support based on the soils and canopy cover within the patch. In this context, carrying capacity serves as a proxy for habitat quality. Gopher tortoises prefer deep sandy soils in which to burrow, and soil types have been previously classified as to their suitability for gopher tortoises. In addition, burrow densities (and presumably tortoise densities) have been documented as varying with soil suitability. Canopy cover also has been shown to exert a strong influence on habitat selection by tortoises and thus is an important component of habitat carrying capacity. 
Patches also were characterized by whether a certain percentage of their area was comprised of the following features: (a) wetlands which serve as barriers to movement by tortoises, (b) active firing ranges and tank maneuver areas (hereafter referred to collectively as ranges) many of which are inaccessible and therefore have not been surveyed or do not have the potential for future gopher tortoise management due to this inaccessibility, and (c) Gopher Tortoise Management Areas (GTMAs) which correspond to delineated areas on Fort Stewart known to support concentrations of gopher tortoises.  The number of successful recruits (egg-to-juvenile stage) also was a patch variable which was based on the number of females that occupied each patch, the probability that a particular female reproduced in that year, and clutch size (which was affected by patch quality). 

## Individual (agent) variables
Agent-based variables were selected and parameterized to represent demographic traits associated with individual gopher tortoises. Individuals were classified as one of two types of agents (juveniles and adults) that behaved differently in the model. Mortality probability, dispersal probability, and dispersal distance varied between juveniles and adults. Juveniles graduated to adult status at age 15 years. Default parameter values were based on published values for gopher tortoises from Fort Stewart or the southeastern Georgia region when possible, or from data collected elsewhere in the species’ range. When published data were not available for a particular parameter, default values were chosen based on values used in previous demographic models developed for gopher tortoises. Additional parameter values were tested to investigate model sensitivity by manipulating a single variable at a time and then comparing model results. 

## Process overview and scheduling
This model proceeded in monthly time steps for the duration of the 100-year simulations. Although 100 years may only represent at most three tortoise generations, our objectives relied on testing a timeframe more realistic to land and wildlife management goals. Based primarily on life history data available for the species in the literature, our model specified that the following processes (see also “Submodels,” Section 3.7) occur each year during the month(s) indicated. An individual can: disperse from or be evicted from its patch and attempt to relocate to another patch (April--October); assess the relative quality of its patch compared to surrounding patches (May); reproduce (June); age (December); and die (January--December). In addition, mortality in the first year of life (egg stage-to-age 1) occurred in September.

## Input
The model inputs were raster and vector geographic information system (GIS) maps developed using the GRASS GIS (http://grass.itc.it). Raster maps were used to initialize patch variables and vector maps were used for visualization purposes. Raster maps included carrying capacity (generated from soils and canopy cover maps), wetlands, GTMAs, ranges, and study area boundaries. Vector maps included streams. The primary model input was projected carrying capacity which was used as a proxy for habitat quality.

##Submodels

Eviction from patch

Adult tortoises were forced to move from their patch when carrying capacity within the patch was exceeded. The model evicted the youngest adult until the patch was again at or below carrying capacity. Once a tortoise was forced to leave a patch, the individual assessed the immediately surrounding eight patches and moved to the patch with the greatest space available (i.e., the greatest difference between carrying capacity and number of adult tortoises currently occupying the patch). If none of the neighboring cells had space available, the evicted tortoises moved to a randomly selected neighboring patch and continued to search for available space. The evicted tortoise could make up to eight attempts to find space available in neighboring patches per monthly time step, corresponding to a maximum cumulative movement distance of 1.2 km per month. The only additional constraint on movement in the eviction submodel was that tortoises, although allowed to move through ranges, could not move through patches that were classified as wetland. Eviction and the resulting search for patches with available space could occur in any month in which tortoise movement occurs (April--October). 

Search for better habitat

Once per year, tortoises had the opportunity to relocate to better habitat. Every May, each individual assessed whether any of the neighboring patches had space available and then compared the habitat quality (i.e., carrying capacity) of those neighboring patches relative to its current patch. If habitat quality was highest in its current patch, the individual remained in the patch. If habitat quality was higher in one or more neighboring patches with space available, it would relocate to the neighboring patch with the highest habitat quality. Habitat quality (i.e., carrying capacity) was maintained as a static feature in our model based on the presumption that current habitat management efforts for gopher tortoises would continue and because tortoises in our model were ejected from a patch when carrying capacity was exceeded, thereby preventing resource depletion. Thus, as long as an individual did not move from its patch, an individual’s associated habitat quality did not change during model simulations. However, other factors in our model elicited movement among patches by tortoises. This submodel provided tortoises the opportunity to respond to a heterogeneous landscape by moving among patches based on habitat quality and resource availability. In reality, gopher tortoises may elect to search for better habitat in any month during the active season. However, to significantly reduce simulation run time, we constrained the model so that this behavior was only allowed to occur in one month (May). 

Dispersal

Gopher tortoises in high-quality habitat have small home ranges generally defined as 1–2 ha for adult males and <0.5 ha for adult females (Diemer 1992; Eubanks, Michener, and Guyer 2003; Smith, Breininger, and Larson 1997). While gopher tortoises occasionally will make long-distance excursions outside their home range, they will return to their core use areas. However, each year a small percentage of tortoises will disperse from their home range and establish a new home range in another location. We incorporated dispersal behavior into the model, assuming that dispersal could be motivated by factors other than habitat quality or carrying capacity of the current patch. In the dispersal submodel, a certain number of juvenile and adult tortoises were randomly selected to disperse based on previously defined dispersal probabilities. Dispersing tortoises oriented in a random direction and searched for patches with space available and occupied by at least one other tortoise. The maximum dispersal distance varied between adults and juveniles, but if individuals could not find an occupied patch within that distance, they were forced to stop. If tortoises encountered wetland patches while dispersing, they were forced to continue searching and move through patches that were not dominated by wetlands. Dispersal could occur during any month from April--October. 

Reproduction

In the reproduction submodel, females produced eggs in June, which is the peak nesting month for gopher tortoises in southern Georgia (Landers, Garner, and McRae 1980). Adult females were randomly selected to reproduce according to the reproductive probability specified in the model. Clutch size varied among females, based on the habitat quality of their patch. Females in high and moderate quality patches produced the average clutch size specified in the model. Females in patches labeled as “low quality” incurred a reproductive penalty by producing two fewer eggs than the average clutch size. Females in unsuitable patches were not allowed to reproduce. Eggs hatched in September. 

Aging

The age of individuals increased one year every December. After that time, individuals of the appropriate age graduated to the next life stage. 

Mortality

A certain percentage of gopher tortoises died in each monthly time step based on stage-specific mortality probabilities for egg-to-age 1, juveniles, and adults. Monthly mortality probabilities were converted from the annual mortality probabilities specified in Table 7 1, assuming mortality could occur with equal probability in any month of the year including when tortoises were relatively inactive. In addition, any tortoises unable to locate suitable habitat patches by October were not allowed to overwinter there and were forced to die in November. Finally, tortoises were not allowed to live more than 100 years in the model. 
 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Demographic sensitivity ALL" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="90"/>
      <value value="91"/>
      <value value="92"/>
      <value value="93"/>
      <value value="94"/>
      <value value="95"/>
      <value value="96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity clutch 100" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity clutch 80" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="960"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity clutch 60" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity clutch 40" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="480"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity clutch 20" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="240"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="4"/>
      <value value="5"/>
      <value value="6"/>
      <value value="7"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity max_age 100" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity max_age 80" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="960"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity max_age 60" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity max_age 40" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="480"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity max_age 20" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="240"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_age">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity female rep 100" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity female rep 80" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="960"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity female rep 60" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity female rep 40" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="480"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity female rep 20" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="240"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="60"/>
      <value value="70"/>
      <value value="80"/>
      <value value="90"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity egg mort 100" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="90"/>
      <value value="91"/>
      <value value="92"/>
      <value value="93"/>
      <value value="94"/>
      <value value="95"/>
      <value value="96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity egg mort 80" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="960"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="90"/>
      <value value="91"/>
      <value value="92"/>
      <value value="93"/>
      <value value="94"/>
      <value value="95"/>
      <value value="96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity egg mort 60" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="90"/>
      <value value="91"/>
      <value value="92"/>
      <value value="93"/>
      <value value="94"/>
      <value value="95"/>
      <value value="96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity egg mort 40" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="480"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="90"/>
      <value value="91"/>
      <value value="92"/>
      <value value="93"/>
      <value value="94"/>
      <value value="95"/>
      <value value="96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity egg mort 20" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="240"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="90"/>
      <value value="91"/>
      <value value="92"/>
      <value value="93"/>
      <value value="94"/>
      <value value="95"/>
      <value value="96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity juv mort 100" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity juv mort 80" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="960"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity juv mort 60" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity juv mort 40" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="480"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity juv mort 20" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="240"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity adult mort 100" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
      <value value="12"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity adult mort 80" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="960"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
      <value value="12"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity adult mort 60" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="720"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
      <value value="12"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity adult mort 40" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="480"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
      <value value="12"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Demographic sensitivity adult mort 20" repetitions="100" runMetricsEveryStep="false">
    <setup>setup-fts-current</setup>
    <go>go</go>
    <timeLimit steps="240"/>
    <metric>count GT with [ age &lt; 1 ]</metric>
    <metric>count GT with [ age &gt;=  1 AND age &lt; 11 ]</metric>
    <metric>count GT with [ age &gt;= 11 AND age &lt; 21 ]</metric>
    <metric>count GT with [ age &gt;= 21 AND age &lt; 31 ]</metric>
    <metric>count GT with [ age &gt;= 31 AND age &lt; 41 ]</metric>
    <metric>count GT with [ age &gt;= 41 AND age &lt; 51 ]</metric>
    <metric>count GT with [ age &gt;= 51 AND age &lt; 61 ]</metric>
    <metric>count GT with [ age &gt;= 61 AND age &lt; 71 ]</metric>
    <metric>count GT with [ age &gt;= 71 AND age &lt; 81 ]</metric>
    <metric>count GT with [ age &gt;= 81 AND age &lt; 91 ]</metric>
    <metric>count GT with [ age &gt;= 91 AND age &lt; 100 ]</metric>
    <metric>count GT with [ age &gt;= 100 ]</metric>
    <metric>total-GT</metric>
    <metric>eggs-per-female</metric>
    <metric>max_age</metric>
    <metric>female-reproduction-probability</metric>
    <metric>egg-to-juvenile-mortality</metric>
    <metric>juvenile-annual-mortality</metric>
    <metric>adult-annual-mortality</metric>
    <enumeratedValueSet variable="Carrying-capacity-modifier">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-pop-size">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Eggs-per-female">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_age">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="female-reproduction-probability">
      <value value="80"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="egg-to-juvenile-mortality">
      <value value="92"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="juvenile-annual-mortality">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="adult-annual-mortality">
      <value value="1.5"/>
      <value value="3"/>
      <value value="6"/>
      <value value="9"/>
      <value value="12"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
